<?php
// ═══════════════════════════════════════════════════════════════════════════
//  SIGEDEC — Funções de acesso e manipulação de dados (JSON)
// ═══════════════════════════════════════════════════════════════════════════

// ─── Autenticação & Sessão ───────────────────────────────────────────────
function isLoggedIn(): bool {
    return isset($_SESSION['sigedec_user']) && !empty($_SESSION['sigedec_user']['role']);
}

function getSystemData(): array {
    $sys = readJSON(SYSTEM_FILE);
    return [
        'admin' => [
            'email'    => $sys['super_admin']['email']    ?? 'superadmin@sigedec.com',
            'password' => $sys['super_admin']['password'] ?? '',
            'name'     => $sys['super_admin']['name']     ?? 'Admin Geral'
        ],
        'churches' => array_values(array_filter(
            array_map('basename', glob(CHURCHES_DIR . '/*', GLOB_ONLYDIR) ?: [])
        ))
    ];
}

function getChurchUsers(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/users.json');
    return empty($data) ? ['users' => []] : $data;
}

function saveChurchUsers(string $churchId, array $users): bool {
    return writeJSON(getChurchPath($churchId) . '/users.json', $users);
}

// ─── URL helpers ──────────────────────────────────────────────────────────
// Sempre retorna a raiz do site — independente do subdirectório do script activo.
function getBaseUrl(): string {
    return defined('SITE_URL') ? rtrim(SITE_URL, '/') : '';
}

// Root-relative — evita 404 de /admin/assets, /church/assets, etc.
function getAssetsUrl(): string { return '/assets'; }

// ─── Flash messages ───────────────────────────────────────────────────────
function flashMessage(string $type, string $msg): void {
    $_SESSION['flash'] = ['type' => $type, 'msg' => $msg];
}

function getFlash(): ?array {
    if (isset($_SESSION['flash'])) { $f = $_SESSION['flash']; unset($_SESSION['flash']); return $f; }
    return null;
}

// ─── Gerar IDs ───────────────────────────────────────────────────────────
function generateId(string $prefix = ''): string {
    return $prefix . '-' . substr(md5(uniqid((string)mt_rand(), true)), 0, 12);
}

// ─── Caminho da igreja ────────────────────────────────────────────────────
function getChurchPath(string $churchId): string {
    $path = CHURCHES_DIR . '/' . $churchId;
    if (!is_dir($path)) mkdir($path, 0755, true);
    return $path;
}

// ─── Igrejas ─────────────────────────────────────────────────────────────
function getAllChurches(): array {
    $dirs = glob(CHURCHES_DIR . '/*', GLOB_ONLYDIR) ?: [];
    $churches = [];
    foreach ($dirs as $dir) {
        $id = basename($dir);
        $info = readJSON($dir . '/info.json');
        if ($info) $churches[$id] = $info;
    }
    return $churches;
}

function saveChurchInfo(string $churchId, array $info): bool {
    return writeJSON(getChurchPath($churchId) . '/info.json', $info);
}

// ─── Trial & Licença ──────────────────────────────────────────────────────
function getLicenseStatus(array $church): array {
    $trialEnd   = $church['trial_ends']   ?? null;
    $licenseEnd = $church['license_ends'] ?? null;
    $now        = time();

    if ($licenseEnd && strtotime($licenseEnd) > $now) {
        return ['status'=>'licensed', 'expires'=>$licenseEnd, 'days'=> (int)ceil((strtotime($licenseEnd)-$now)/86400)];
    }
    if ($trialEnd) {
        $diff = (int)ceil((strtotime($trialEnd) - $now) / 86400);
        if ($diff > 0) return ['status'=>'trial', 'expires'=>$trialEnd, 'days'=>$diff];
        return ['status'=>'expired', 'expires'=>$trialEnd, 'days'=>0];
    }
    return ['status'=>'expired', 'expires'=>null, 'days'=>0];
}

function isAccessAllowed(array $church): bool {
    $ls = getLicenseStatus($church);
    return in_array($ls['status'], ['trial','licensed']);
}

// ─── Chaves de Licença ────────────────────────────────────────────────────
function getLicenseKeys(): array {
    $data = readJSON(DATA_DIR . '/licenses.json');
    return empty($data) ? ['keys' => []] : $data;
}

function saveLicenseKeys(array $data): bool {
    return writeJSON(DATA_DIR . '/licenses.json', $data);
}

function generateLicenseKey(string $type): string {
    $prefix = strtoupper($type);
    $body   = strtoupper(substr(md5(uniqid((string)mt_rand(), true)), 0, 16));
    return $prefix . '-' . chunk_split($body, 4, '-');
}

function applyLicenseKey(string $churchId, string $key): array {
    $keysData = getLicenseKeys();
    foreach ($keysData['keys'] as &$k) {
        if ($k['key'] === $key) {
            if ($k['used']) return ['ok'=>false,'msg'=>'Esta chave já foi utilizada.'];
            $church = getChurchInfo($churchId);
            if (!$church)   return ['ok'=>false,'msg'=>'Igreja não encontrada.'];

            $months = match($k['type']) { '1yr'=>12, '2yr'=>24, '3yr'=>36, 'trial'=>6, default=>6 };
            $from = max(time(), strtotime($church['license_ends'] ?? 'now'));
            $until = date('Y-m-d', strtotime("+{$months} months", $from));

            if ($k['type'] === 'trial') {
                $church['trial_ends'] = $until;
            } else {
                $church['license_ends'] = $until;
            }
            $k['used']       = true;
            $k['used_by']    = $churchId;
            $k['used_at']    = date('Y-m-d H:i:s');
            saveChurchInfo($churchId, $church);
            saveLicenseKeys($keysData);
            return ['ok'=>true,'msg'=>'Licença activada até '.$until.'.'];
        }
    }
    return ['ok'=>false,'msg'=>'Chave inválida ou não encontrada.'];
}

// ─── Membros ──────────────────────────────────────────────────────────────
function getMembers(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/members.json');
    return empty($data) ? ['members' => []] : $data;
}

function saveMembers(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/members.json', $data);
}

function getMemberById(string $churchId, string $memberId): ?array {
    foreach (getMembers($churchId)['members'] as $m) {
        if ($m['id'] === $memberId) return $m;
    }
    return null;
}

function getMemberName(string $churchId, ?string $memberId): string {
    if (!$memberId) return '—';
    $m = getMemberById($churchId, $memberId);
    return $m ? $m['name'] : '—';
}

function saveMember(string $churchId, array $member): bool {
    $data = getMembers($churchId);
    $found = false;
    foreach ($data['members'] as &$m) {
        if ($m['id'] === $member['id']) { $m = $member; $found = true; break; }
    }
    if (!$found) $data['members'][] = $member;
    return saveMembers($churchId, $data);
}

// ─── Lista negra ──────────────────────────────────────────────────────────
function getBlacklist(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/blacklist.json');
    return empty($data) ? ['members' => []] : $data;
}

function saveBlacklist(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/blacklist.json', $data);
}

// ─── Subdivisões ──────────────────────────────────────────────────────────
function getSubdivisions(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/subdivisions.json');
    return empty($data) ? ['subdivisions' => [], 'denomination_terms' => []] : $data;
}

function saveSubdivisions(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/subdivisions.json', $data);
}

function getSubdivisionById(string $churchId, string $subId): ?array {
    foreach (getSubdivisions($churchId)['subdivisions'] as $s) {
        if ($s['id'] === $subId) return $s;
    }
    return null;
}

function getSubdivisionName(string $churchId, ?string $subId): string {
    if (!$subId) return '—';
    $s = getSubdivisionById($churchId, $subId);
    return $s ? $s['name'] : '—';
}

// ─── Ministérios ──────────────────────────────────────────────────────────
function getMinistries(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/ministries.json');
    return empty($data) ? ['types' => [], 'assignments' => []] : $data;
}

function saveMinistries(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/ministries.json', $data);
}

// ─── Líderes ─────────────────────────────────────────────────────────────
function getLeaders(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/leaders.json');
    return empty($data) ? ['leaders' => []] : $data;
}

function saveLeaders(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/leaders.json', $data);
}

// ─── Financeiro ──────────────────────────────────────────────────────────
function getFinance(string $churchId): array {
    $data = readJSON(getChurchPath($churchId) . '/finance.json');
    return empty($data) ? ['categories' => [], 'transactions' => []] : $data;
}

function saveFinance(string $churchId, array $data): bool {
    return writeJSON(getChurchPath($churchId) . '/finance.json', $data);
}

// ─── Labels & Utilitários ────────────────────────────────────────────────
function statusLabel(string $status): string {
    return match($status) { 'activo'=>'Activo','cortado'=>'Cortado','desvinculado'=>'Desvinculado', default=>ucfirst($status) };
}

function statusBadge(string $status): string {
    return match($status) { 'activo'=>'badge-success','cortado'=>'badge-warning','desvinculado'=>'badge-danger', default=>'badge-grey' };
}

function statusDot(string $status): string {
    return match($status) { 'activo'=>'#22c55e','cortado'=>'#f59e0b','desvinculado'=>'#ef4444', default=>'#94a3b8' };
}

function maritalLabel(string $status): string {
    return match($status) { 'solteiro'=>'Solteiro(a)','casado'=>'Casado(a)','viuvo'=>'Viúvo(a)','divorciado'=>'Divorciado(a)', default=>ucfirst($status) };
}

function age(?string $birthDate): string {
    if (!$birthDate) return '—';
    try { return (new DateTime($birthDate))->diff(new DateTime())->y . ' anos'; } catch (Exception $e) { return '—'; }
}

function calcTotals(array $transactions, string $type): float {
    return array_sum(array_map(fn($t) => ($t['type']===$type) ? (float)($t['amount']??0) : 0, $transactions));
}

function isStrongPassword(string $pass): bool {
    return strlen($pass) >= 6
        && preg_match('/[A-Za-z]/', $pass)
        && preg_match('/[0-9]/', $pass)
        && preg_match('/[^A-Za-z0-9]/', $pass);
}


// ─── Branding da Igreja (cores + logotipo) ──────────────────────────────
function getChurchBranding(string $churchId): array {
    $info = getChurchInfo($churchId) ?? [];
    return [
        'primary'  => $info['brand_primary']  ?? '#0f1729',   // azul-marinho padrão
        'accent'   => $info['brand_accent']   ?? '#c9a84c',   // dourado padrão
        'logo_b64' => $info['logo_b64']       ?? '',          // base64 do logotipo
        'logo_type'=> $info['logo_type']      ?? '',          // image/png, image/jpeg...
    ];
}

function saveChurchBranding(string $churchId, array $branding): bool {
    $info = getChurchInfo($churchId) ?? [];
    if (!empty($branding['primary']))  $info['brand_primary']  = $branding['primary'];
    if (!empty($branding['accent']))   $info['brand_accent']   = $branding['accent'];
    if (isset($branding['logo_b64']))  $info['logo_b64']       = $branding['logo_b64'];
    if (isset($branding['logo_type'])) $info['logo_type']      = $branding['logo_type'];
    return saveChurchInfo($churchId, $info);
}

function getLogoUrl(array $branding): string {
    if (!empty($branding['logo_b64']) && !empty($branding['logo_type'])) {
        return 'data:' . $branding['logo_type'] . ';base64,' . $branding['logo_b64'];
    }
    return '';
}

// ─── Dízimo / Contribuições ───────────────────────────────────────────────
function titheLabel(string $status, string $lang = 'pt'): string {
    if ($lang === 'en') {
        return match($status) {
            'regular'   => 'Regular',
            'irregular' => 'Irregular',
            'infiel'    => 'Unfaithful',
            default     => '—'
        };
    }
    return match($status) {
        'regular'   => 'Regular',
        'irregular' => 'Irregular',
        'infiel'    => 'Infiel',
        default     => '—'
    };
}

function titheBadge(string $status): string {
    return match($status) {
        'regular'   => 'badge-success',
        'irregular' => 'badge-warning',
        'infiel'    => 'badge-danger',
        default     => 'badge-grey'
    };
}

// ─── Portal do Membro ─────────────────────────────────────────────────────
function findMemberByPortalEmail(string $email): ?array {
    $email = strtolower(trim($email));
    foreach (getAllChurches() as $cid => $church) {
        $data = getMembers($cid);
        foreach ($data['members'] as $m) {
            if (!empty($m['portal_email']) && strtolower($m['portal_email']) === $email) {
                return ['member' => $m, 'church_id' => $cid, 'church' => $church];
            }
        }
    }
    return null;
}

function getMemberPhotoUrl(string $churchId, string $memberId): ?string {
    $exts = ['jpg','jpeg','png','webp'];
    foreach ($exts as $ext) {
        $file = getChurchPath($churchId) . '/photos/' . $memberId . '.' . $ext;
        if (file_exists($file)) {
            // Use photo proxy to avoid exposing the data directory
            return '/public/photo.php?church=' . urlencode($churchId) . '&member=' . urlencode($memberId);
        }
    }
    return null;
}

function saveMemberPhoto(string $churchId, string $memberId, string $base64Data): bool|string {
    if (preg_match('/^data:image\/(\w+);base64,/', $base64Data, $matches)) {
        $type = strtolower($matches[1]);
        if (!in_array($type, ['jpg','jpeg','png','webp'])) $type = 'jpg';
        $base64Data = preg_replace('/^data:image\/\w+;base64,/', '', $base64Data);
    } else {
        $type = 'jpg';
    }
    $photoDir = getChurchPath($churchId) . '/photos';
    if (!is_dir($photoDir)) mkdir($photoDir, 0755, true);
    $data = base64_decode($base64Data);
    if (!$data) return 'Erro ao decodificar imagem.';
    if (strlen($data) > 2 * 1024 * 1024) return 'A imagem excede 2 MB.';
    // Remove old photos
    foreach (['jpg','jpeg','png','webp'] as $ext) {
        $old = $photoDir . '/' . $memberId . '.' . $ext;
        if (file_exists($old)) unlink($old);
    }
    $path = $photoDir . '/' . $memberId . '.' . $type;
    return file_put_contents($path, $data) !== false ? true : 'Erro ao guardar a imagem.';
}
